let activeContact = null;
let socket = null;

// Initialize Socket.IO connection
function initSocket() {
    socket = io();

    socket.on('connect', () => {
        console.log('Connected to server');
    });

    // Listen for new messages
    socket.on('chat-message', (msg) => {
        // Only display if this message is for the currently active contact
        if (activeContact && 
            (msg.sender_id === activeContact.id || msg.receiver_id === activeContact.id)) {
            displayMessage(msg);
        }
    });

    // Listen for new message notifications (to update contacts list if needed)
    socket.on('new-message', (data) => {
        // Optionally refresh contacts or show notification
        console.log('New message from contact:', data.contactId);
    });
}

// Display a single message in the chat
function displayMessage(msg) {
    const msgBox = document.getElementById("messages");
    const myId = 1; // Admin user ID
    
    const isSent = msg.sender_id === myId;
    const div = document.createElement('div');
    div.className = "message " + (isSent ? "sent" : "received");
    div.innerText = msg.body;
    msgBox.appendChild(div);
    msgBox.scrollTop = msgBox.scrollHeight;
}

// Load contacts
async function loadContacts() {
    const res = await fetch('/api/contacts');
    const contacts = await res.json();

    const div = document.getElementById('contacts');
    div.innerHTML = "";

    contacts.forEach(c => {
        const el = document.createElement('div');
        el.className = "contact";
        el.innerText = c.name;
        el.onclick = () => openChat(c);
        div.appendChild(el);
    });
}

// Open chat window
async function openChat(contact) {
    activeContact = contact;
    document.getElementById("contactName").innerText = contact.name;

    // Join the socket room for this contact
    if (socket) {
        socket.emit('join-room', contact.id);
    }

    const res = await fetch(`/api/messages/${contact.id}`);
    const msgs = await res.json();

    const msgBox = document.getElementById("messages");
    msgBox.innerHTML = "";

    const myId = 1; // Admin user ID

    msgs.forEach(m => {
        const isSent = m.sender_id === myId;
        let div = document.createElement('div');
        div.className = "message " + (isSent ? "sent" : "received");
        div.innerText = m.body;
        msgBox.appendChild(div);
    });

    msgBox.scrollTop = msgBox.scrollHeight;
}

// Send a message
document.getElementById("sendBtn").onclick = async () => {
    if (!activeContact) return;

    const text = document.getElementById("messageInput").value;
    if (!text.trim()) return;

    const response = await fetch('/send', {
        method: "POST",
        headers: {"Content-Type": "application/json"},
        body: JSON.stringify({
            contact_id: activeContact.id,
            text: text
        })
    });

    const data = await response.json();
    
    // Message will be displayed via socket event, but we can also display it immediately
    if (data.success && data.message) {
        displayMessage(data.message);
    }

    document.getElementById("messageInput").value = "";
};

// Initialize socket and load contacts when page loads
initSocket();
loadContacts();
